<?php
/**
 * MultiChat WHMCS Integration Module
 * 
 * Integrates MultiChat live chat system with WHMCS
 * Allows viewing client information from chat conversations
 * Displays chat widget in client area
 */

use WHMCS\Database\Capsule;

if (!defined("WHMCS")) {
    die("This file cannot be accessed directly");
}

/**
 * Module Configuration
 */
function multichat_config() {
    return [
        'name' => 'MultiChat Integration',
        'description' => 'Integrates MultiChat live chat system with WHMCS for seamless customer support',
        'version' => '1.0',
        'author' => 'Hired Hosting',
        'language' => 'english',
        'fields' => [
            'multichat_url' => [
                'FriendlyName' => 'MultiChat API URL',
                'Type' => 'text',
                'Size' => '50',
                'Default' => 'https://multichat.website',
                'Description' => 'Your MultiChat installation URL (e.g., https://multichat.website)',
            ],
            'api_key' => [
                'FriendlyName' => 'API Key',
                'Type' => 'password',
                'Size' => '50',
                'Default' => '',
                'Description' => 'WHMCS API Key from MultiChat (find at bottom of members/settings.php)',
            ],
            'bubble_key' => [
                'FriendlyName' => 'Chat Bubble Key',
                'Type' => 'text',
                'Size' => '50',
                'Default' => '',
                'Description' => 'Chat bubble key from MultiChat for widget integration',
            ],
            'enable_client_portal' => [
                'FriendlyName' => 'Enable Client Portal Widget',
                'Type' => 'yesno',
                'Description' => 'Display chat widget in client area',
                'Default' => 'yes',
            ],
            'auto_link_customers' => [
                'FriendlyName' => 'Auto-Link Customers',
                'Type' => 'yesno',
                'Description' => 'Automatically sync WHMCS clients to MultiChat',
                'Default' => 'yes',
            ],
            'show_in_dashboard' => [
                'FriendlyName' => 'Show in Admin Dashboard',
                'Type' => 'yesno',
                'Description' => 'Display MultiChat stats in admin dashboard',
                'Default' => 'yes',
            ],
        ]
    ];
}

/**
 * Activate Module
 * Creates database table for storing client-customer links
 */
function multichat_activate() {
    try {
        if (!Capsule::schema()->hasTable('mod_multichat_clients')) {
            Capsule::schema()->create('mod_multichat_clients', function ($table) {
                $table->increments('id');
                $table->integer('client_id')->unique();
                $table->integer('multichat_customer_id');
                $table->timestamp('last_sync')->nullable();
                $table->index('client_id');
            });
        }
        
        return [
            'status' => 'success',
            'description' => 'MultiChat Integration module activated successfully!',
        ];
    } catch (Exception $e) {
        return [
            'status' => 'error',
            'description' => 'Unable to create database table: ' . $e->getMessage(),
        ];
    }
}

/**
 * Deactivate Module
 */
function multichat_deactivate() {
    return [
        'status' => 'success',
        'description' => 'MultiChat Integration module deactivated. Database table preserved.',
    ];
}

/**
 * Upgrade Module
 */
function multichat_upgrade($vars) {
    $currentVersion = $vars['version'];
    
    // Future upgrade logic here
    
    return;
}

/**
 * Admin Dashboard Output
 */
function multichat_output($vars) {
    $modulelink = $vars['modulelink'];
    $version = $vars['version'];
    $LANG = $vars['_lang'];
    
    // Get statistics
    $stats = multichat_get_stats($vars);
    
    // Display output
    echo '<div class="container-fluid">';
    echo '<h2>MultiChat Integration</h2>';
    echo '<p class="text-muted">Manage your MultiChat live chat integration with WHMCS</p>';
    
    // Action Buttons
    echo '<div class="row mb-4">';
    echo '<div class="col-md-12">';
    
    if (!empty($vars['multichat_url'])) {
        $dashboardUrl = rtrim($vars['multichat_url'], '/') . '/members/app-dashboard.php';
        echo '<a href="' . $dashboardUrl . '" target="_blank" class="btn btn-info"><i class="fas fa-external-link-alt"></i> Open MultiChat Dashboard</a>';
    }
    echo '</div>';
    echo '</div>';
    
    // Auto-sync info
    echo '<div class="alert alert-info" style="margin: 20px 0;">';
    echo '<i class="fas fa-info-circle"></i> <strong>Automatic Sync:</strong> Customers are automatically synced to MultiChat when they log in to the client area.';
    echo '</div>';
    
    // Statistics Cards
    echo '<div class="row">';
    
    // Total WHMCS Clients
    echo '<div class="col-md-4 mb-3">';
    echo '<div class="card">';
    echo '<div class="card-body">';
    echo '<h5 class="card-title text-muted">WHMCS Clients</h5>';
    echo '<h2 class="mb-0">' . number_format($stats['total_whmcs_clients']) . '</h2>';
    echo '</div>';
    echo '</div>';
    echo '</div>';
    
    // MultiChat Customers
    if ($stats['multichat_stats']) {
        echo '<div class="col-md-4 mb-3">';
        echo '<div class="card border-info">';
        echo '<div class="card-body">';
        echo '<h5 class="card-title text-info">MultiChat Customers</h5>';
        echo '<h2 class="mb-0 text-info">' . number_format($stats['multichat_stats']['total_customers']) . '</h2>';
        echo '</div>';
        echo '</div>';
        echo '</div>';
        
        // Active Conversations
        echo '<div class="col-md-4 mb-3">';
        echo '<div class="card border-warning">';
        echo '<div class="card-body">';
        echo '<h5 class="card-title text-warning">Active Chats</h5>';
        echo '<h2 class="mb-0 text-warning">' . number_format($stats['multichat_stats']['active_conversations']) . '</h2>';
        echo '</div>';
        echo '</div>';
        echo '</div>';
    }
    
    echo '</div>'; // End row
    
    // Configuration Status
    echo '<div class="card mt-4">';
    echo '<div class="card-header"><h5 class="mb-0">Configuration Status</h5></div>';
    echo '<div class="card-body">';
    echo '<table class="table table-borderless">';
    echo '<tr><td><strong>MultiChat URL:</strong></td><td>' . (!empty($vars['multichat_url']) ? '<span class="text-success">✓ Configured</span>' : '<span class="text-danger">✗ Not Set</span>') . '</td></tr>';
    echo '<tr><td><strong>API Key:</strong></td><td>' . (!empty($vars['api_key']) ? '<span class="text-success">✓ Configured</span>' : '<span class="text-danger">✗ Not Set</span>') . '</td></tr>';
    echo '<tr><td><strong>Bubble Key:</strong></td><td>' . (!empty($vars['bubble_key']) ? '<span class="text-success">✓ Configured</span>' : '<span class="text-danger">✗ Not Set</span>') . '</td></tr>';
    echo '<tr><td><strong>Client Portal Widget:</strong></td><td>' . ($vars['enable_client_portal'] == 'on' ? '<span class="text-success">Enabled</span>' : '<span class="text-muted">Disabled</span>') . '</td></tr>';
    echo '<tr><td><strong>Auto-Link Customers:</strong></td><td>' . ($vars['auto_link_customers'] == 'on' ? '<span class="text-success">Enabled</span>' : '<span class="text-muted">Disabled</span>') . '</td></tr>';
    echo '</table>';
    echo '</div>';
    echo '</div>';
    
    echo '</div>'; // End container
}

/**
 * Client Area Output
 */
function multichat_clientarea($vars) {
    if ($vars['enable_client_portal'] != 'on') {
        return [];
    }
    
    $client = Menu::context('client');
    $multichatUrl = rtrim($vars['multichat_url'], '/');
    $bubbleKey = $vars['bubble_key'];
    
    // Prepare client data
    $clientId = $client ? $client->id : '';
    $clientEmail = $client ? $client->email : '';
    $clientName = $client ? $client->fullName : '';
    
    return [
        'pagetitle' => 'Live Chat Support',
        'breadcrumb' => ['index.php?m=multichat' => 'Live Chat Support'],
        'templatefile' => 'clientarea',
        'requirelogin' => true,
        'vars' => [
            'multichat_url' => $multichatUrl,
            'bubble_key' => $bubbleKey,
            'client_id' => $clientId,
            'client_email' => $clientEmail,
            'client_name' => $clientName,
        ],
    ];
}

/**
 * Test API Connection
 */
function multichat_test_connection($vars) {
    try {
        // Validate configuration
        if (empty($vars['multichat_url'])) {
            return [
                'success' => false,
                'message' => 'MultiChat URL is not configured. Please configure it in module settings.',
            ];
        }
        
        if (empty($vars['api_key'])) {
            return [
                'success' => false,
                'message' => 'API Key is not configured. Please configure it in module settings.',
            ];
        }
        
        if (empty($vars['bubble_key'])) {
            return [
                'success' => false,
                'message' => 'Bubble Key is not configured. Please configure it in module settings.',
            ];
        }
        
        // Test the API endpoint
        $apiUrl = rtrim($vars['multichat_url'], '/') . '/app/whmcs_api.php?action=test';
        
        logActivity('MultiChat: Testing API connection to ' . $apiUrl);
        
        $ch = curl_init($apiUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $vars['api_key'],
            'Content-Type: application/json',
        ]);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);
        
        if ($curlError) {
            logActivity('MultiChat Error: cURL Error - ' . $curlError);
            return [
                'success' => false,
                'message' => 'Connection error: ' . $curlError,
            ];
        }
        
        logActivity('MultiChat: HTTP Response Code: ' . $httpCode);
        logActivity('MultiChat: Response: ' . substr($response, 0, 200));
        
        if ($httpCode == 401) {
            return [
                'success' => false,
                'message' => 'API Key is invalid. Please verify your API key at: ' . rtrim($vars['multichat_url'], '/') . '/members/whmcs_api_keys.php',
            ];
        }
        
        if ($httpCode == 404) {
            return [
                'success' => false,
                'message' => 'API endpoint not found. Please verify the MultiChat URL is correct.',
            ];
        }
        
        if ($httpCode != 200) {
            return [
                'success' => false,
                'message' => 'API returned HTTP ' . $httpCode . '. Response: ' . substr($response, 0, 100),
            ];
        }
        
        // Try to decode JSON response
        $data = json_decode($response, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            // Not JSON but got 200, might still be working
            logActivity('MultiChat: Non-JSON response but HTTP 200 - considering success');
            return [
                'success' => true,
                'message' => 'Connection successful! API is responding (HTTP 200). Bubble Key: ' . substr($vars['bubble_key'], 0, 10) . '...',
            ];
        }
        
        // Check response format
        if (isset($data['success']) && $data['success']) {
            logActivity('MultiChat: API test successful');
            return [
                'success' => true,
                'message' => 'Connection successful! API authentication verified. Your MultiChat integration is working properly.',
            ];
        } else {
            $errorMsg = isset($data['message']) ? $data['message'] : 'Unknown response';
            logActivity('MultiChat: API test failed - ' . $errorMsg);
            return [
                'success' => false,
                'message' => 'API responded but returned error: ' . $errorMsg,
            ];
        }
        
    } catch (Exception $e) {
        logActivity('MultiChat Error: ' . $e->getMessage());
        return [
            'success' => false,
            'message' => 'Error: ' . $e->getMessage(),
        ];
    }
}

/**
 * Sync All Customers
 */
function multichat_sync_all_customers($vars) {
    try {
        $clients = Capsule::table('tblclients')
            ->where('status', 'Active')
            ->get();
        
        $synced = 0;
        $failed = 0;
        
        foreach ($clients as $client) {
            $result = multichat_sync_customer($client->id, $vars);
            if ($result['success']) {
                $synced++;
            } else {
                $failed++;
            }
        }
        
        return [
            'success' => true,
            'message' => "Sync completed! Successfully synced: {$synced}, Failed: {$failed}",
        ];
    } catch (Exception $e) {
        return [
            'success' => false,
            'message' => 'Sync failed: ' . $e->getMessage(),
        ];
    }
}

/**
 * Get Statistics
 */
function multichat_get_stats($vars) {
    $totalClients = Capsule::table('tblclients')->count();
    $linkedCustomers = Capsule::table('mod_multichat_clients')->count();
    $syncPercentage = $totalClients > 0 ? round(($linkedCustomers / $totalClients) * 100, 1) : 0;
    
    // Get MultiChat stats
    $multichatStats = null;
    try {
        $response = multichat_api_call('stats', [], 'GET', $vars);
        if ($response && isset($response['success']) && $response['success']) {
            $multichatStats = $response['stats'];
        }
    } catch (Exception $e) {
        // Ignore errors
    }
    
    return [
        'total_whmcs_clients' => $totalClients,
        'linked_customers' => $linkedCustomers,
        'sync_percentage' => $syncPercentage,
        'multichat_stats' => $multichatStats,
    ];
}

/**
 * Sync Single Customer
 */
function multichat_sync_customer($clientId, $vars) {
    try {
        $client = Capsule::table('tblclients')->where('id', $clientId)->first();
        
        if (!$client) {
            return ['success' => false, 'message' => 'Client not found'];
        }
        
        // Get client's services/products
        $services = Capsule::table('tblhosting')
            ->where('userid', $clientId)
            ->get();
        
        $productList = [];
        foreach ($services as $service) {
            // Get product name
            $product = Capsule::table('tblproducts')
                ->where('id', $service->packageid)
                ->first();
            
            $productList[] = [
                'service_id' => $service->id,
                'product_id' => $service->packageid,
                'product_name' => $product ? $product->name : 'Unknown Product',
                'domain' => $service->domain,
                'status' => $service->domainstatus,
                'billing_cycle' => $service->billingcycle,
                'amount' => $service->amount,
                'next_due_date' => $service->nextduedate,
                'registration_date' => $service->regdate,
            ];
        }
        
        // Get client's invoices
        $invoices = Capsule::table('tblinvoices')
            ->where('userid', $clientId)
            ->orderBy('id', 'desc')
            ->limit(10)
            ->get();
        
        $invoiceList = [];
        foreach ($invoices as $invoice) {
            $invoiceList[] = [
                'invoice_id' => $invoice->id,
                'date' => $invoice->date,
                'due_date' => $invoice->duedate,
                'total' => $invoice->total,
                'status' => $invoice->status,
            ];
        }
        
        // Get client's tickets
        $tickets = Capsule::table('tbltickets')
            ->where('userid', $clientId)
            ->orderBy('id', 'desc')
            ->limit(10)
            ->get();
        
        $ticketList = [];
        foreach ($tickets as $ticket) {
            $ticketList[] = [
                'ticket_id' => $ticket->id,
                'title' => $ticket->title,
                'status' => $ticket->status,
                'priority' => $ticket->urgency,
                'date' => $ticket->date,
            ];
        }
        
        $data = [
            'email' => $client->email,
            'name' => $client->firstname . ' ' . $client->lastname,
            'whmcs_client_id' => $clientId,
            'phone' => $client->phonenumber,
            'company' => $client->companyname,
            'address' => $client->address1,
            'city' => $client->city,
            'state' => $client->state,
            'postcode' => $client->postcode,
            'country' => $client->country,
            'status' => $client->status,
            'currency' => $client->currency,
            'products' => $productList,
            'invoices' => $invoiceList,
            'tickets' => $ticketList,
            'total_services' => count($productList),
            'credit' => $client->credit,
        ];
        
        $response = multichat_api_call('sync_customer', $data, 'POST', $vars);
        
        if ($response && isset($response['success']) && $response['success']) {
            // Update local database
            Capsule::table('mod_multichat_clients')->updateOrInsert(
                ['client_id' => $clientId],
                [
                    'multichat_customer_id' => $response['customer_id'],
                    'last_sync' => date('Y-m-d H:i:s'),
                ]
            );
            
            return ['success' => true, 'customer_id' => $response['customer_id']];
        } else {
            return ['success' => false, 'message' => isset($response['message']) ? $response['message'] : 'Sync failed'];
        }
    } catch (Exception $e) {
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

/**
 * Make API Call to MultiChat
 */
function multichat_api_call($endpoint, $data = [], $method = 'GET', $vars = null) {
    if (!$vars) {
        $vars = getModuleSettings('multichat');
    }
    
    $apiUrl = rtrim($vars['multichat_url'], '/');
    $apiKey = $vars['api_key'];
    
    if (empty($apiUrl)) {
        logActivity('MultiChat Error: API URL not configured');
        throw new Exception('MultiChat URL not configured');
    }
    
    if (empty($apiKey)) {
        logActivity('MultiChat Error: API Key not configured');
        throw new Exception('API Key not configured');
    }
    
    $url = $apiUrl . '/app/whmcs_api.php?action=' . $endpoint;
    
    logActivity('MultiChat: Making API call to ' . $url);
    
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bearer ' . $apiKey,
        'Content-Type: application/json',
    ]);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    if ($method === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    }
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    logActivity('MultiChat: HTTP Response Code: ' . $httpCode);
    logActivity('MultiChat: Raw Response: ' . substr($response, 0, 500));
    
    if ($error) {
        logActivity('MultiChat Error: cURL Error - ' . $error);
        throw new Exception('cURL Error: ' . $error);
    }
    
    if ($httpCode == 401) {
        logActivity('MultiChat Error: HTTP 401 - Invalid API Key');
        throw new Exception('Invalid or missing API key. Please verify your API key in module settings and ensure it\'s correctly generated in MultiChat at: members/whmcs_api_keys.php');
    }
    
    if ($httpCode != 200) {
        logActivity('MultiChat Error: HTTP ' . $httpCode . ' - Response: ' . $response);
        throw new Exception('HTTP Error ' . $httpCode . ': ' . $response);
    }
    
    $decoded = json_decode($response, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        logActivity('MultiChat Error: JSON decode error - ' . json_last_error_msg());
        throw new Exception('Invalid JSON response: ' . json_last_error_msg());
    }
    
    return $decoded;
}

/**
 * Get Module Settings Helper
 */
function getModuleSettings($moduleName) {
    $settings = [];
    $results = Capsule::table('tbladdonmodules')
        ->where('module', $moduleName)
        ->get();
    
    foreach ($results as $result) {
        $settings[$result->setting] = $result->value;
    }
    
    return $settings;
}
